
program plotspec3

  ! written by: jochen autschbach, SUNY Buffalo, NY, USA
  !             email: jochena@buffalo.edu
  
  ! output graph data with oscillator and rotatory strengths
  ! obtained from ADF cd-spectrum computation.
  ! each excitation is overlayed with a Gauss function, and
  ! multiplied by a factor such as to yield the R and f
  ! values for each single transition from the integration
  ! over the curve

  ! a script "parse-cd3" is used to extract the data from the ADF
  ! output. The namelist and a blank line is automatically inserted
  ! at the beginning of the "parse-cd3" output.
  ! input options with defaults are as follows:

  !  sigma = 0 ! use calculated (empirical) linewidth, otherwise
  !            ! sigma is given explicitly if nonzero 
  !  sharpen = 1.0d0 ! "sharpen" the linewidth by a factor 1/sharpen
  !                  ! (only for sigma=0)
  !  npoints = 200 ! no. of points to generate for the plot
  !  nexcit = 0 ! default no. of excitations to plot (determined by input
  !             ! or by symmetry info if zero
  !  border = 3d0 ! add this many linewidths at the plot boundaries
  !  lorentz = .f. ! use lorentz curve for lineshape instead of gauss
  !  old = .f. ! compatibility with old buggy implementation:
  !            ! R's were off by factor of two but otherwise correct
  !  invert = .f. ! invert CD spectrum

  ! the program assumes input of excitation energies in eV, oscillator
  ! strengths in a.u., and rotatory strengths in 10^(-40)
  ! c.g.s. units

  implicit double precision (a-h,o-z)

  double precision, dimension (:,:), allocatable :: results
  integer, dimension(100) :: nsyme, ndegen
  data nsyme/ 100*0/
  data ndegen/ 100*1/
  dimension temp(9)
  parameter one=1d0, zero=0d0, two=2d0, four=4d0, pi=3.14159265359d0
  parameter ev2cm = 8065.48d0
  logical old,lorentz,invert, magnify
  dimension window(2)
  data window /0d0,0d0/

  ! the namelist plot contains the input options. see above for defaults
  
  namelist /plot/ npoints, sigma, nsyme, nexcit, sharpen, eshift, &
     &            old, border, lorentz, ndegen, invert, window,   &
     &            magnification, exponent

  ! each excitation gets assigned a normalized Gauss curve, centered at x0.
  ! alternatively, lorentz curves can be used.
  ! (x0 is the excitation energy in eV):

  gauss(x,x0,sigma) = 1/(sqrt(two*pi)*sigma)*exp(-(x-x0)**2 /(two * sigma**2))
  glorentz(x,x0,sigma) = (sigma / (two* pi))/ &
     & ((x - x0)**2 + (sigma/two)**2)

  ! See CP 224 (1997), 143-155 for the conversion factors. please email me
  ! if you really need your simulated spectrum with more more than 3 or 4
  ! significant digits of accuracy and you happen to have the conversion
  ! factors at hand

  rinteg(x0) = 22.97d0 / x0
  finteg(x0) =  3.48d-2 

  ! See JACS 126 (2004), 1408 for the following conversion factor (exept we
  ! have devided by sqrt(2) because of the factor of 1/2 we have in the 
  ! gaussian exponent

  delta_eps(x,x0,sigma)= (1.737d-2 / sigma)*exp(-(x-x0)**2 /(two * sigma**2))

  ! empirical linewidth from Brown et al., J. Chem. Soc (A) 1971, 751
  ! in eV, using 1eV approx. 8066/cm here:

  sigma1(x0) = 1d0/ev2cm * 7.5d0 * sqrt(x0 * ev2cm)

  logical lcalsig, oldrconv

  ! =====================================================================

  write (6,'(/a/)') ' P L O T S P E C'

  open (7,file='impulses.dat',status='unknown')
  open (8,file='graph.dat',status='unknown')

  ! plot defaults. some can be changed by namelist input
  eshift = 0.d0 ! shift energies in the spectrum (not used)
  sigma = 0.0d0 ! use calculated (empirical) linewidth if zero
  lcalsig = .true.
  old = .false. ! compatibility with old buggy implementation:
                ! R's were off by factor of two but otherwise correct
  sharpen = 1.0d0 ! "sharpen" the linewidth by a factor 1/sharpen
  npoints = 200
  nexcit = 0 ! default no. of excitations to plot (determined by input
             ! or by symmetry info (ALL)
  border = 3d0 ! add that many sigmas at the plot boundaries
  lorentz = .false. ! use lorentz curve for lineshape instead of gauss
  invert = .false. ! invert the CD spectrum
  magnify = .false. ! magnify a certain energy window
  magnification = 10 ! default magnification
  oldrconv = .true.

  ! how many data points do we have? no. of plot points, sigma?
  ! read the namelist "plot" from input file:
  read (5,plot)

  if (sigma.ne.0d0) then
    lcalsig = .false.
    if (sigma.le.0d0) then
      write (0,*) 'linewidth < 0 requested. aborting ...'
      close (7,status='delete')
      close (8,status='delete')
      stop 'error termination'
    end if
    write (6,'(1X,a,f5.3,a)') 'using fixed linewidth of ',sigma,' eV'
  else
    write (6,'(1X,a)') 'using empirical formula for linewidth' 
  end if
  rfac = one
  if (old) then
    rfac = two
    write(6,'(1X,a,f3.0)') &
       &'Rotatory strengths will be multiplied by',rfac
  endif

  ! lorentz curves are not broad enough if compared to gaussians
  ! with same sigma parameter. 
  if (lorentz) then
    sigma = two*sigma
    sharpen = sharpen/two
  end if

  if (invert) then
    rinvert = -one
  else
    rinvert = one
  end if

  ! magnify certain portion of the plot
  if (window(2).gt.zero) then
    if (window(1).gt.window(2)) stop 'window in error'
    magnify = .true.
  end if

  ! how many symmetries ?
  nsym = 0
  i = 0
  10 continue
  i = i+1
  if (nsyme(i).gt.0) then
    nsym = nsym + 1
    goto 10
  endif
  if (nsym.le.0) then
    write (0,*) 'error nsyme: no entries found. aborting ...'
    close (7,status='delete')
    close (8,status='delete')
    stop 'error termination'
  endif

  np = 0
  do i=1,nsym
    np = np + nsyme(i)
  enddo
  if (np.le.0) then
    write (0,*) 'error np: nothing to plot. aborting ...'
    close (7,status='delete')
    close (8,status='delete')
    stop 'error termination'
  endif

  if (nexcit.eq.0) then ! no input value for nexcit, set it to
    nexcit = np         ! its default = plot all excitations
  else ! check that nexcit is not too large
    if (nexcit.gt.np) then
      write (0,*) 'error: nexcit is greater than no. of energies. aborting'
      close (7,status='delete')
      close (8,status='delete')
      stop 'error termination'
    endif
  endif

  write (6,'(1X,a,i4,a,i4,a)') 'using ',nexcit,' out of ',np, &
     &' excitations for the graph data'

  allocate (results(np,9))
  results = 0

  write (7,*) '# E/eV, f / a.u., R / 10**40 cgs'

  ll = 1
  do isym=1,nsym
!    write (0,*) 'symmetry',isym
    read (5,'(//////)') 
    
    ! read the energies, osclillator strengths
    do i=ll,ll+nsyme(isym)-1
      read (5,*) ndum, rdum,results(i,1:3)
      !    write (6,*) ndum,results(i,1:3)
    enddo
    
    read (5,'(//////)')
    
    ! read the rotational strengths
    do i=ll,ll+nsyme(isym)-1
      read (5,*) ndum, results(i,6:9)
      !    write (6,*) ndum,results(i,6:9)
    enddo

    ! skip data blocks belonging to dipole velocity representation
    read (5,'(/)')

!!$    read (5,'(//////)')
!!$    do i=ll,ll+nsyme(isym)-1
!!$      read (5,*)
!!$    enddo
!!$    read (5,'(/////)')
!!$        do i=ll,ll+nsyme(isym)-1
!!$      read (5,*)
!!$    enddo
    
    ! write "impulses" data
     
    write (7,*) '# Symmetry ',isym
    do i=ll,ll+nsyme(isym)-1

      ! multiply UV and CD intensities with degeneracy of irrep
      results(i,2) = results(i,2) * float(ndegen(isym))
      results(i,6) = results(i,6) * float(ndegen(isym)) * rinvert

      !                    energy       oscil.       rotstr.
      write (7,'(3e15.7)') results(i,1),results(i,2),rfac*results(i,6)
    enddo
    write (7,*) ; write (7,*)

    ll = ll+ nsyme(isym)
  enddo ! isym

  ! sort data in ascending energy (sum up all symmetries)
  ! (probably not a fast algorithm, but it seems to work...)

  do k=2,np
    ll = k-1
    small = results(ll,1)
    do i=k,np
      if (results(i,1).lt. small) then
!        write (6,*) k,i,small, results(i,1)
        temp(:) = results(ll,:)
        results(ll,:) = results(i,:)
        results(i,:) = temp(:)
        small = results(ll,1)
      endif
    enddo
  enddo

  do i=2,np
    if (results(i,1).lt.results(i-1,1)) then 
      write (0,*) results(1:np,1)
      write (0,*) 'sorting error, aborting...'
      close (7,status='delete')
      close (8,status='delete')
      stop 'error termination'
    endif
  enddo

  ! determine plot range:
  emin = results(1,1)
  emax = results(nexcit,1) ! it's nexcit here, not np 

  !  write (6,*) emin,emax

  ! add  border*sigma of approx. 0.15 eV at the energy boundaries for plot:
  emin = emin - border * 0.15d0
  emax = emax + border * 0.15d0

  ! number of plot points defines step size deltae:
  deltae = abs(emax-emin)/float(npoints-1)
  
!  write (6,*) emin,emax,deltae

  write (8,*) '# E/eV, epsilon/10**(-3)l/(mol cm) , delta epsilon / l/(mol cm)'
  do i=0,npoints
    fval = 0.d0
    rval = 0.d0
    x = emin + float(i) * deltae
    do j=1,np
      eexcit = results(j,1)
      oscstr = results(j,2)
      rotstr = rfac*results(j,6)
      if (lcalsig) sigma = sigma1(eexcit)/sharpen ! compute linewidth
      x0 = results(j,1) ! excitation energy in eV = center of the curve
      if (.not.lorentz) then
        if (oldrconv) then
          rval = rval + rotstr*gauss(x,x0,sigma)/rinteg(x0)
        else
          rval = rval + rotstr*delta_eps(x,x0,sigma)*x0
        end if

        fval = fval + oscstr*gauss(x,x0,sigma)/finteg(x0)
      else
        rval = rval + rotstr*glorentz(x,x0,sigma)/rinteg(x0)
        fval = fval + oscstr*glorentz(x,x0,sigma)/finteg(x0)
      end if
    enddo

! magnify portion of the spectrum 
    if (magnify) then
      if (x.ge.window(1) .and. x.lt.window(2)) then
      write (8,'(4e15.7)') x,fval,rval,float(magnification)*rval
      else
      write (8,'(3e15.7)') x,fval,rval
      end if
    else
    write (8,'(3e15.7)') x,fval,rval
    end if
  enddo

  ! we are done
  deallocate (results)
  close (7,status='keep')
  close (8,status='keep')

  write (6,*) 'Note: use the script "graphit" to plot the data'
  stop 'normal termination'
  end
